"""
Define the unit tests for the
:mod:`colour_checker_detection.detection.segmentation` module.
"""

from __future__ import annotations

import glob
import os
import platform

import numpy as np
import pytest
from colour import read_image

from colour_checker_detection import ROOT_RESOURCES_TESTS
from colour_checker_detection.detection import (
    detect_colour_checkers_segmentation,
    extractor_segmentation,
    segmenter_default,
)

__author__ = "Colour Developers"
__copyright__ = "Copyright 2018 Colour Developers"
__license__ = "BSD-3-Clause - https://opensource.org/licenses/BSD-3-Clause"
__maintainer__ = "Colour Developers"
__email__ = "colour-developers@colour-science.org"
__status__ = "Production"

__all__ = [
    "DETECTION_DIRECTORY",
    "PNG_FILES",
    "TestSegmenterDefault",
    "TestExtractorSegmentation",
    "TestDetectColourCheckersSegmentation",
]

DETECTION_DIRECTORY = os.path.join(
    ROOT_RESOURCES_TESTS, "colour_checker_detection", "detection"
)

PNG_FILES = sorted(glob.glob(os.path.join(DETECTION_DIRECTORY, "IMG_19*.png")))


class TestSegmenterDefault:
    """
    Define :func:`colour_checker_detection.detection.segmentation.\
segmenter_default` definition unit tests methods.
    """

    @pytest.mark.skipif(
        platform.system() in ("Windows", "Microsoft", "Linux"),
        reason="Unit test is only reproducible on macOS",
    )
    def test_segmenter_default(self) -> None:
        """
        Define :func:`colour_checker_detection.detection.segmentation.\
segmenter_default` definition unit tests methods.
        """

        # Skipping unit test when "png" files are missing, e.g., when testing
        # the distributed "Python" package.
        if len(PNG_FILES) == 0:
            return

        colour_checkers_rectangles = [
            # IMG_1966.png
            np.array([[[671, 578], [675, 364], [994, 370], [990, 584]]]),
            # IMG_1967.png
            np.array([[[357, 691], [372, 219], [1086, 242], [1071, 713]]]),
            # IMG_1968.png
            np.array([[[572, 670], [576, 357], [1045, 363], [1041, 676]]]),
            # IMG_1969.png
            np.array([[[616, 605], [617, 310], [1056, 312], [1055, 607]]]),
            # IMG_1970.png
            np.array([[[639, 333], [795, 333], [795, 437], [639, 437]]]),
            # IMG_1971.png
            np.array([[[759, 654], [762, 288], [1010, 290], [1007, 657]]]),
        ]

        for i, png_file in enumerate(PNG_FILES):
            np.testing.assert_array_equal(
                segmenter_default(read_image(png_file), additional_data=False),
                colour_checkers_rectangles[i],
            )

        (
            colour_checkers,
            clusters,
            swatches,
            segmented_image,
        ) = segmenter_default(read_image(PNG_FILES[0]), additional_data=True).values

        np.testing.assert_array_equal(
            colour_checkers,
            colour_checkers_rectangles[0],
        )

        np.testing.assert_array_equal(
            clusters,
            np.array([[[671, 578], [675, 364], [994, 370], [990, 584]]]),
        )

        np.testing.assert_array_equal(
            swatches,
            np.array(
                [
                    [[886, 575], [887, 531], [931, 532], [930, 576]],
                    [[835, 574], [836, 530], [880, 531], [879, 575]],
                    [[782, 572], [784, 528], [829, 530], [828, 573]],
                    [[732, 571], [732, 527], [777, 529], [776, 572]],
                    [[681, 570], [682, 526], [726, 527], [725, 571]],
                    [[939, 525], [940, 481], [984, 483], [982, 526]],
                    [[887, 524], [887, 479], [933, 481], [932, 525]],
                    [[835, 523], [836, 478], [881, 480], [879, 524]],
                    [[784, 522], [784, 477], [829, 478], [829, 523]],
                    [[733, 521], [733, 477], [778, 479], [777, 522]],
                    [[683, 518], [684, 478], [726, 479], [725, 519]],
                    [[939, 474], [939, 429], [986, 431], [985, 475]],
                    [[888, 473], [889, 429], [934, 430], [933, 474]],
                    [[837, 472], [837, 428], [882, 430], [881, 473]],
                    [[785, 472], [785, 427], [831, 429], [830, 473]],
                    [[734, 470], [734, 426], [779, 428], [778, 471]],
                    [[682, 469], [683, 425], [728, 427], [727, 471]],
                    [[941, 423], [941, 378], [987, 380], [986, 424]],
                    [[889, 422], [889, 377], [935, 379], [934, 423]],
                    [[838, 421], [838, 377], [883, 379], [882, 422]],
                    [[786, 420], [786, 376], [831, 378], [830, 421]],
                    [[735, 420], [736, 376], [780, 377], [779, 421]],
                    [[683, 419], [684, 375], [728, 376], [727, 420]],
                ],
            ),
        )

        np.testing.assert_array_equal(
            segmented_image.shape,
            (959, 1440),
        )


class TestExtractorSegmentation:
    """Define :func:`extractor_segmentation` definition unit tests methods."""

    def test_extractor_segmentation(self) -> None:
        """Test :func:`extractor_segmentation` definition."""

        # Test the extractor function with segmentation data
        image = read_image(PNG_FILES[0])

        # First get segmentation data
        segmentation_data = segmenter_default(image, additional_data=True)

        # Then use extractor to get colors
        extractors_result = extractor_segmentation(
            image, segmentation_data, additional_data=True
        )

        # Should return tuple of DataDetectionColourChecker
        assert isinstance(extractors_result, tuple)
        assert len(extractors_result) == 1

        # Check that result has expected attributes
        colour_checker_data = extractors_result[0]
        assert hasattr(colour_checker_data, "swatch_colours")
        assert hasattr(colour_checker_data, "colour_checker")
        assert hasattr(colour_checker_data, "swatch_masks")

        # Check shape of swatch colors (24 swatches, RGB)
        assert colour_checker_data.swatch_colours.shape == (24, 3)

        # Test without additional_data
        extractors_result_simple = extractor_segmentation(
            image, segmentation_data, additional_data=False
        )

        # Should return tuple of NDArrayFloat when additional_data=False
        assert isinstance(extractors_result_simple, tuple)
        assert len(extractors_result_simple) == 1

        # When additional_data=False, should return just the swatch colors
        assert isinstance(extractors_result_simple[0], np.ndarray)
        assert extractors_result_simple[0].shape == (24, 3)

        # Colors should be similar between both calls
        np.testing.assert_allclose(
            extractors_result[0].swatch_colours,
            extractors_result_simple[0],
            atol=0.0001,
        )


class TestDetectColourCheckersSegmentation:
    """
    Define :func:`colour_checker_detection.detection.segmentation.\
detect_colour_checkers_segmentation` definition unit tests methods.
    """

    @pytest.mark.skipif(
        platform.system() in ("Windows", "Microsoft", "Linux"),
        reason="Unit test is only reproducible on macOS",
    )
    def test_detect_colour_checkers_segmentation(self) -> None:
        """
        Define :func:`colour_checker_detection.detection.segmentation.\
detect_colour_checkers_segmentation` definition unit tests methods.
        """

        # Skipping unit test when "png" files are missing, e.g., when testing
        # the distributed "Python" package.
        if len(PNG_FILES) == 0:
            return

        test_swatches = [
            # IMG_1966.png
            (
                np.array(
                    [
                        [0.24832383, 0.15328871, 0.08390528],
                        [0.41702515, 0.25749645, 0.15524487],
                        [0.21561632, 0.19969580, 0.18122308],
                        [0.19740795, 0.17287970, 0.06432715],
                        [0.27344641, 0.20384602, 0.19483535],
                        [0.23397270, 0.28900447, 0.18982877],
                        [0.45813105, 0.23316218, 0.05019258],
                        [0.18077615, 0.16044180, 0.20741640],
                        [0.39589152, 0.16688587, 0.11324922],
                        [0.19540377, 0.11008169, 0.11556859],
                        [0.31544343, 0.29114881, 0.02143807],
                        [0.43209532, 0.25869009, 0.00787305],
                        [0.13758199, 0.12513705, 0.17688017],
                        [0.18623975, 0.24561603, 0.07675898],
                        [0.35410595, 0.13109548, 0.07380733],
                        [0.46201408, 0.32264870, 0.00161150],
                        [0.37113905, 0.15819806, 0.16307077],
                        [0.11848511, 0.20048647, 0.18547055],
                        [0.50230730, 0.41071278, 0.28996512],
                        [0.41354588, 0.33473289, 0.23615897],
                        [0.32904309, 0.26473305, 0.18369149],
                        [0.23612075, 0.18612471, 0.12523490],
                        [0.16774438, 0.13109615, 0.08313587],
                        [0.10746267, 0.07903581, 0.04796651],
                    ],
                ),
            ),
            # IMG_1967.png
            (
                np.array(
                    [
                        [0.36018878, 0.22291452, 0.11730091],
                        [0.62564981, 0.39444405, 0.24182397],
                        [0.33206907, 0.31609666, 0.28861040],
                        [0.30452022, 0.27339727, 0.10480344],
                        [0.41740695, 0.31912389, 0.30785984],
                        [0.34865320, 0.43936506, 0.29126465],
                        [0.67974621, 0.35227019, 0.06983876],
                        [0.27157846, 0.25354025, 0.33056694],
                        [0.62124234, 0.27033532, 0.18669768],
                        [0.30703250, 0.17978220, 0.19183952],
                        [0.48548090, 0.45865157, 0.03294417],
                        [0.65081471, 0.40023121, 0.01611003],
                        [0.19301628, 0.18572472, 0.27450651],
                        [0.28079671, 0.38511795, 0.12274225],
                        [0.55472660, 0.21451427, 0.12551409],
                        [0.72074521, 0.51505238, 0.00542995],
                        [0.57743728, 0.25776306, 0.26855013],
                        [0.17295605, 0.31638905, 0.29510689],
                        [0.73900223, 0.60931826, 0.43826103],
                        [0.62775826, 0.51768881, 0.37181169],
                        [0.51389074, 0.42036685, 0.29863322],
                        [0.36946711, 0.30227602, 0.20832038],
                        [0.26305991, 0.21490613, 0.14286397],
                        [0.16101658, 0.13380753, 0.08050805],
                    ],
                ),
            ),
            # IMG_1968.png
            (
                np.array(
                    [
                        [0.34663522, 0.22224304, 0.12062097],
                        [0.58243281, 0.37286037, 0.23490584],
                        [0.31593052, 0.30380291, 0.27724549],
                        [0.30289200, 0.27467039, 0.11844290],
                        [0.42093673, 0.32420260, 0.30901504],
                        [0.35755947, 0.43322891, 0.29118398],
                        [0.62269628, 0.32643992, 0.07461109],
                        [0.25821927, 0.23577160, 0.30442065],
                        [0.57680899, 0.25524464, 0.17697753],
                        [0.29972863, 0.18389925, 0.18966803],
                        [0.48380157, 0.45361868, 0.05204606],
                        [0.64555794, 0.39815736, 0.02867807],
                        [0.18776426, 0.17572677, 0.24990779],
                        [0.26354364, 0.35200754, 0.11395297],
                        [0.51605159, 0.20282018, 0.11844741],
                        [0.68874329, 0.49213710, 0.00387706],
                        [0.56534988, 0.25589401, 0.26516667],
                        [0.19274078, 0.31693918, 0.29429474],
                        [0.69554168, 0.56929833, 0.40149987],
                        [0.58874923, 0.48029220, 0.34048769],
                        [0.48317575, 0.39269075, 0.27530921],
                        [0.35990325, 0.28928345, 0.19722588],
                        [0.26885545, 0.21622665, 0.14608587],
                        [0.18056251, 0.14151992, 0.09341659],
                    ],
                ),
            ),
            # IMG_1969.png
            (
                np.array(
                    [
                        [0.21938626, 0.12745625, 0.05788294],
                        [0.38446504, 0.23450573, 0.13715135],
                        [0.19894665, 0.18729475, 0.17246075],
                        [0.19457348, 0.17182909, 0.06108426],
                        [0.27911529, 0.21303453, 0.20818117],
                        [0.24418873, 0.30463862, 0.20364815],
                        [0.42587042, 0.21036983, 0.03387973],
                        [0.16159414, 0.14428590, 0.19156292],
                        [0.39223763, 0.16027561, 0.10546608],
                        [0.19678101, 0.11048739, 0.11770960],
                        [0.33779100, 0.31829613, 0.02421810],
                        [0.48373029, 0.29524031, 0.01024585],
                        [0.11387850, 0.10201313, 0.15105926],
                        [0.17331047, 0.22999455, 0.06060657],
                        [0.35458463, 0.12845676, 0.06968934],
                        [0.48692709, 0.34049782, 0.00180221],
                        [0.40718290, 0.17617357, 0.18679525],
                        [0.13581324, 0.23464102, 0.22059691],
                        [0.48099062, 0.37892416, 0.25402120],
                        [0.41220644, 0.32620439, 0.22248316],
                        [0.34211776, 0.27174953, 0.18414445],
                        [0.25703460, 0.20097046, 0.13262190],
                        [0.19677229, 0.15521792, 0.10373505],
                        [0.13552929, 0.10757796, 0.06661499],
                    ],
                ),
            ),
            # IMG_1970.png
            (
                np.array(
                    [
                        [0.35696816, 0.21930856, 0.11403693],
                        [0.60821688, 0.38355675, 0.23785689],
                        [0.31905308, 0.30316707, 0.27976862],
                        [0.30302203, 0.27013326, 0.10675076],
                        [0.42698881, 0.33104691, 0.32175022],
                        [0.37930816, 0.47598192, 0.32185996],
                        [0.66308343, 0.34331018, 0.04840572],
                        [0.25116241, 0.23189382, 0.31125751],
                        [0.59939820, 0.25615332, 0.17492208],
                        [0.29469383, 0.16788641, 0.18995292],
                        [0.49911827, 0.47258803, 0.02826829],
                        [0.68877065, 0.43108177, 0.00497005],
                        [0.18456070, 0.16787593, 0.25497437],
                        [0.27599725, 0.36154932, 0.11149685],
                        [0.53841883, 0.20049028, 0.12016004],
                        [0.72244722, 0.51131302, 0.00018977],
                        [0.57876974, 0.25831616, 0.27366117],
                        [0.17160977, 0.33335912, 0.31680411],
                        [0.76642007, 0.61226165, 0.42404577],
                        [0.63404596, 0.51218498, 0.36258361],
                        [0.51259273, 0.41126823, 0.28810981],
                        [0.38513795, 0.30572629, 0.20889392],
                        [0.27351546, 0.21982883, 0.15018983],
                        [0.17083941, 0.13807549, 0.08634300],
                    ],
                ),
            ),
            # IMG_1971.png
            (
                np.array(
                    [
                        [0.52386993, 0.33313319, 0.18489836],
                        [0.88912624, 0.57027519, 0.36043468],
                        [0.47387072, 0.45559943, 0.42352596],
                        [0.44801432, 0.40738437, 0.16351445],
                        [0.61939651, 0.47873873, 0.47075850],
                        [0.53648233, 0.66336131, 0.45412394],
                        [0.94693035, 0.50383645, 0.09576066],
                        [0.36924785, 0.35085207, 0.46480927],
                        [0.86583030, 0.38429213, 0.27107581],
                        [0.43026441, 0.25643602, 0.28056529],
                        [0.70629638, 0.67313057, 0.04883015],
                        [0.96674675, 0.60350400, 0.01688687],
                        [0.27080789, 0.25053880, 0.37763739],
                        [0.39222810, 0.52593189, 0.17089139],
                        [0.76616114, 0.29511422, 0.18367355],
                        [0.99521071, 0.72431517, 0.00122845],
                        [0.81165427, 0.36755443, 0.39317939],
                        [0.23630118, 0.46475750, 0.44472364],
                        [0.99935937, 0.84607446, 0.60752183],
                        [0.87117517, 0.71787977, 0.51672125],
                        [0.70453340, 0.58058864, 0.41745698],
                        [0.52914387, 0.43000436, 0.30116317],
                        [0.37784788, 0.30666250, 0.21504804],
                        [0.23438329, 0.18970679, 0.12232202],
                    ],
                ),
            ),
        ]

        for i, png_file in enumerate(PNG_FILES):
            np.testing.assert_allclose(
                detect_colour_checkers_segmentation(png_file, additional_data=False),
                test_swatches[i],
                atol=0.0001,
            )

        (
            swatch_colours,
            swatch_masks,
            colour_checker,
            quadrilateral,
        ) = detect_colour_checkers_segmentation(
            read_image(PNG_FILES[0]), additional_data=True
        )[0].values

        np.testing.assert_allclose(
            swatch_colours,
            test_swatches[0][0],
            atol=0.0001,
        )

        np.testing.assert_array_equal(
            colour_checker.shape[0:2],
            np.array([960, 1440]),
        )

        np.testing.assert_array_equal(
            swatch_masks,
            np.array(
                [
                    [104, 136, 104, 136],
                    [104, 136, 344, 376],
                    [104, 136, 584, 616],
                    [104, 136, 824, 856],
                    [104, 136, 1064, 1096],
                    [104, 136, 1304, 1336],
                    [344, 376, 104, 136],
                    [344, 376, 344, 376],
                    [344, 376, 584, 616],
                    [344, 376, 824, 856],
                    [344, 376, 1064, 1096],
                    [344, 376, 1304, 1336],
                    [584, 616, 104, 136],
                    [584, 616, 344, 376],
                    [584, 616, 584, 616],
                    [584, 616, 824, 856],
                    [584, 616, 1064, 1096],
                    [584, 616, 1304, 1336],
                    [824, 856, 104, 136],
                    [824, 856, 344, 376],
                    [824, 856, 584, 616],
                    [824, 856, 824, 856],
                    [824, 856, 1064, 1096],
                    [824, 856, 1304, 1336],
                ]
            ),
        )

        assert quadrilateral.shape == (4, 2)
