/**
 Copyright 2013 BlackBerry Inc.
 Copyright (c) 2013-2016 Chukong Technologies Inc.
 Copyright (c) 2017-2023 Xiamen Yaji Software Co., Ltd.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 Original file from GamePlay3D: http://gameplay3d.org

 This file was modified to fit the cocos2d-x project
 */

#include "math/Vec4.h"

#include <cmath>
#include "base/Macros.h"
#include "math/MathUtil.h"

#if CC_PLATFORM != CC_PLATFORM_EMSCRIPTEN
    #include "base/std/hash/hash.h"
#endif

NS_CC_MATH_BEGIN

Vec4::Vec4()
: x(0.0F),
  y(0.0F),
  z(0.0F),
  w(0.0F) {
}

Vec4::Vec4(float xx, float yy, float zz, float ww)
: x(xx),
  y(yy),
  z(zz),
  w(ww) {
}

Vec4::Vec4(const float *src) {
    set(src);
}

Vec4::Vec4(const Vec4 &p1, const Vec4 &p2) {
    set(p1, p2);
}

Vec4::Vec4(const Vec4 &copy) {
    set(copy);
}

Vec4 Vec4::fromColor(unsigned int color) {
    float components[4];
    int componentIndex = 0;
    for (int i = 3; i >= 0; --i) {
        uint32_t component = (color >> i * 8) & 0x000000ff;

        components[componentIndex++] = static_cast<float>(component) / 255.0F;
    }

    Vec4 value(components);
    return value;
}

bool Vec4::isZero() const {
    return x == 0.0F && y == 0.0F && z == 0.0F && w == 0.0F;
}

bool Vec4::isOne() const {
    return x == 1.0F && y == 1.0F && z == 1.0F && w == 1.0F;
}

float Vec4::angle(const Vec4 &v1, const Vec4 &v2) {
    const float dx = (v1.y * v2.z - v1.z * v2.y);
    const float dy = (v1.z * v2.x - v1.x * v2.z);
    const float dz = (v1.x * v2.y - v1.y * v2.x);
    const float dotVal = (v1.x * v2.x + v1.y * v2.y + v1.z * v2.z);

    return std::atan2(std::sqrt(dx * dx + dy * dy + dz * dz) + MATH_FLOAT_SMALL, dotVal);
}

void Vec4::add(const Vec4 &v) {
    x += v.x;
    y += v.y;
    z += v.z;
    w += v.w;
}

void Vec4::add(const Vec4 &v1, const Vec4 &v2, Vec4 *dst) {
    CC_ASSERT(dst);

    dst->x = v1.x + v2.x;
    dst->y = v1.y + v2.y;
    dst->z = v1.z + v2.z;
    dst->w = v1.w + v2.w;
}

void Vec4::clamp(const Vec4 &min, const Vec4 &max) {
    CC_ASSERT(!(min.x > max.x || min.y > max.y || min.z > max.z || min.w > max.w));

    // Clamp the x value.
    if (x < min.x) {
        x = min.x;
    }
    if (x > max.x) {
        x = max.x;
    }

    // Clamp the y value.
    if (y < min.y) {
        y = min.y;
    }
    if (y > max.y) {
        y = max.y;
    }

    // Clamp the z value.
    if (z < min.z) {
        z = min.z;
    }
    if (z > max.z) {
        z = max.z;
    }

    // Clamp the z value.
    if (w < min.w) {
        w = min.w;
    }
    if (w > max.w) {
        w = max.w;
    }
}

void Vec4::clamp(const Vec4 &v, const Vec4 &min, const Vec4 &max, Vec4 *dst) {
    CC_ASSERT(dst);
    CC_ASSERT(!(min.x > max.x || min.y > max.y || min.z > max.z || min.w > max.w));

    // Clamp the x value.
    dst->x = v.x;
    if (dst->x < min.x) {
        dst->x = min.x;
    }
    if (dst->x > max.x) {
        dst->x = max.x;
    }

    // Clamp the y value.
    dst->y = v.y;
    if (dst->y < min.y) {
        dst->y = min.y;
    }
    if (dst->y > max.y) {
        dst->y = max.y;
    }

    // Clamp the z value.
    dst->z = v.z;
    if (dst->z < min.z) {
        dst->z = min.z;
    }
    if (dst->z > max.z) {
        dst->z = max.z;
    }

    // Clamp the w value.
    dst->w = v.w;
    if (dst->w < min.w) {
        dst->w = min.w;
    }
    if (dst->w > max.w) {
        dst->w = max.w;
    }
}

float Vec4::distance(const Vec4 &v) const {
    float dx = v.x - x;
    float dy = v.y - y;
    float dz = v.z - z;
    float dw = v.w - w;

    return std::sqrt(dx * dx + dy * dy + dz * dz + dw * dw);
}

float Vec4::distanceSquared(const Vec4 &v) const {
    float dx = v.x - x;
    float dy = v.y - y;
    float dz = v.z - z;
    float dw = v.w - w;

    return (dx * dx + dy * dy + dz * dz + dw * dw);
}

float Vec4::dot(const Vec4 &v) const {
    return (x * v.x + y * v.y + z * v.z + w * v.w);
}

float Vec4::dot(const Vec4 &v1, const Vec4 &v2) {
    return (v1.x * v2.x + v1.y * v2.y + v1.z * v2.z + v1.w * v2.w);
}

float Vec4::length() const {
    return std::sqrt(x * x + y * y + z * z + w * w);
}

float Vec4::lengthSquared() const {
    return (x * x + y * y + z * z + w * w);
}

void Vec4::negate() {
    x = -x;
    y = -y;
    z = -z;
    w = -w;
}

/**
* The inverse value of this vector.
* 
* This method set each component to its inverse value, zero
* will become infinity.
*/
void Vec4::inverse(const Vec4 &v, Vec4 *dst) {
    CC_ASSERT(dst);

    dst->x = 1.0F / v.x;
    dst->y = 1.0F / v.y;
    dst->z = 1.0F / v.z;
    dst->w = 1.0F / v.w;
}

void Vec4::normalize() {
    float n = x * x + y * y + z * z + w * w;
    // Already normalized.
    if (n == 1.0F) {
        return;
    }

    n = std::sqrt(n);
    // Too close to zero.
    if (n < MATH_TOLERANCE) {
        return;
    }

    n = 1.0F / n;
    x *= n;
    y *= n;
    z *= n;
    w *= n;
}

Vec4 Vec4::getNormalized() const {
    Vec4 v(*this);
    v.normalize();
    return v;
}

void Vec4::scale(float scalar) {
    x *= scalar;
    y *= scalar;
    z *= scalar;
    w *= scalar;
}

void Vec4::set(float xx, float yy, float zz, float ww) {
    this->x = xx;
    this->y = yy;
    this->z = zz;
    this->w = ww;
}

void Vec4::set(const float *array) {
    CC_ASSERT(array);

    x = array[0];
    y = array[1];
    z = array[2];
    w = array[3];
}

void Vec4::set(const Vec4 &v) {
    this->x = v.x;
    this->y = v.y;
    this->z = v.z;
    this->w = v.w;
}

void Vec4::set(const Vec4 &p1, const Vec4 &p2) {
    x = p2.x - p1.x;
    y = p2.y - p1.y;
    z = p2.z - p1.z;
    w = p2.w - p1.w;
}

void Vec4::subtract(const Vec4 &v) {
    x -= v.x;
    y -= v.y;
    z -= v.z;
    w -= v.w;
}

void Vec4::subtract(const Vec4 &v1, const Vec4 &v2, Vec4 *dst) {
    CC_ASSERT(dst);

    dst->x = v1.x - v2.x;
    dst->y = v1.y - v2.y;
    dst->z = v1.z - v2.z;
    dst->w = v1.w - v2.w;
}

void Vec4::lerp(const Vec4 &a, const Vec4 &b, float t, Vec4 *dst) {
    CC_ASSERT(dst);
    dst->x = a.x + t * (b.x - a.x);
    dst->y = a.y + t * (b.y - a.y);
    dst->z = a.z + t * (b.z - a.z);
    dst->w = a.w + t * (b.w - a.w);
}

const Vec4 Vec4::ZERO = Vec4(0.0F, 0.0F, 0.0F, 0.0F);
const Vec4 Vec4::ONE = Vec4(1.0F, 1.0F, 1.0F, 1.0F);
const Vec4 Vec4::UNIT_X = Vec4(1.0F, 0.0F, 0.0F, 0.0F);
const Vec4 Vec4::UNIT_Y = Vec4(0.0F, 1.0F, 0.0F, 0.0F);
const Vec4 Vec4::UNIT_Z = Vec4(0.0F, 0.0F, 1.0F, 0.0F);
const Vec4 Vec4::UNIT_W = Vec4(0.0F, 0.0F, 0.0F, 1.0F);

#if CC_PLATFORM != CC_PLATFORM_EMSCRIPTEN
template <>
ccstd::hash_t Hasher<Vec4>::operator()(const Vec4 &v) const {
    return ccstd::hash_range(reinterpret_cast<const uint64_t *>(&v.x),
                             reinterpret_cast<const uint64_t *>(&v.x + 4));
}
#endif

NS_CC_MATH_END
