/*
 * Copyright (C) 2005-2014 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

//
// C/C++ logging functions.  See the logging documentation for API details.
//
// We'd like these to be available from C code (in case we import some from
// somewhere), so this has a C interface.
//
// The output will be correct when the log file is shared between multiple
// threads and/or multiple processes so long as the operating system
// supports O_APPEND.  These calls have mutex-protected data structures
// and so are NOT reentrant.  Do not use LOG in a signal handler.
//
#ifndef COCOS_CUTILS_LOG_H
#define COCOS_CUTILS_LOG_H

#include <stdarg.h>
#include <stdio.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>
#if CC_PLATFORM == CC_PLATFORM_ANDROID
#include <android/log.h>
#elif CC_PLATFORM == CC_PLATFORM_OPENHARMONY
#include <hilog/log.h>
#define LOG_VERBOSE LOG_INFO
#endif

#ifdef __cplusplus
extern "C" {
#endif

// ---------------------------------------------------------------------
/*
 * Normally we strip ALOGV (VERBOSE messages) from release builds.
 * You can modify this (for example with "#define LOG_NDEBUG 0"
 * at the top of your source file) to change that behavior.
 */
#ifndef LOG_NDEBUG
    #if defined(CC_DEBUG) && CC_DEBUG > 0
        #define LOG_NDEBUG 0
    #else
        #define LOG_NDEBUG 1
    #endif
#endif

/*
 * This is the local tag used for the following simplified
 * logging macros.  You can change this preprocessor definition
 * before using the other macros to change the tag.
 */
#ifndef LOG_TAG
    #define LOG_TAG NULL
#endif

// ---------------------------------------------------------------------

#ifndef __predict_false
    #define __predict_false(exp) __builtin_expect((exp) != 0, 0)
#endif

/*
 *      -DLINT_RLOG in sources that you want to enforce that all logging
 * goes to the radio log buffer. If any logging goes to any of the other
 * log buffers, there will be a compile or link error to highlight the
 * problem. This is not a replacement for a full audit of the code since
 * this only catches compiled code, not ifdef'd debug code. Options to
 * defining this, either temporarily to do a spot check, or permanently
 * to enforce, in all the communications trees; We have hopes to ensure
 * that by supplying just the radio log buffer that the communications
 * teams will have their one-stop shop for triaging issues.
 */
#ifndef LINT_RLOG

    /*
 * Simplified macro to send a verbose log message using the current LOG_TAG.
 */
    #ifndef ALOGV
#if CC_PLATFORM == CC_PLATFORM_ANDROID
    #define __ALOGV(...) ((void)ALOG(LOG_VERBOSE, LOG_TAG, __VA_ARGS__))
#elif CC_PLATFORM == CC_PLATFORM_OPENHARMONY
    #define __ALOGV(...) ((void)ALOG(LOG_INFO, LOG_TAG, __VA_ARGS__))
#endif
        #if LOG_NDEBUG
            #define ALOGV(...)                \
                do {                          \
                    if (0) {                  \
                        __ALOGV(__VA_ARGS__); \
                    }                         \
                } while (0)
        #else
            #define ALOGV(...) __ALOGV(__VA_ARGS__)
        #endif
    #endif

    #ifndef ALOGV_IF
        #if LOG_NDEBUG
            #define ALOGV_IF(cond, ...) ((void)0)
        #else
            #define ALOGV_IF(cond, ...)                                \
                ((__predict_false(cond))                               \
                     ? ((void)ALOG(LOG_VERBOSE, LOG_TAG, __VA_ARGS__)) \
                     : (void)0)
        #endif
    #endif

    /*
 * Simplified macro to send a debug log message using the current LOG_TAG.
 */
    #ifndef ALOGD
        #define ALOGD(...) ((void)ALOG(LOG_DEBUG, LOG_TAG, __VA_ARGS__))
    #endif

    #ifndef ALOGD_IF
        #define ALOGD_IF(cond, ...)                              \
            ((__predict_false(cond))                             \
                 ? ((void)ALOG(LOG_DEBUG, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif

    /*
 * Simplified macro to send an info log message using the current LOG_TAG.
 */
    #ifndef ALOGI
        #define ALOGI(...) ((void)ALOG(LOG_INFO, LOG_TAG, __VA_ARGS__))
    #endif

    #ifndef ALOGI_IF
        #define ALOGI_IF(cond, ...)                             \
            ((__predict_false(cond))                            \
                 ? ((void)ALOG(LOG_INFO, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif

    /*
 * Simplified macro to send a warning log message using the current LOG_TAG.
 */
    #ifndef ALOGW
        #define ALOGW(...) ((void)ALOG(LOG_WARN, LOG_TAG, __VA_ARGS__))
    #endif

    #ifndef ALOGW_IF
        #define ALOGW_IF(cond, ...)                             \
            ((__predict_false(cond))                            \
                 ? ((void)ALOG(LOG_WARN, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif

    /*
 * Simplified macro to send an error log message using the current LOG_TAG.
 */
    #ifndef ALOGE
        #define ALOGE(...) ((void)ALOG(LOG_ERROR, LOG_TAG, __VA_ARGS__))
    #endif

    #ifndef ALOGE_IF
        #define ALOGE_IF(cond, ...)                              \
            ((__predict_false(cond))                             \
                 ? ((void)ALOG(LOG_ERROR, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif

    // ---------------------------------------------------------------------

    /*
 * Conditional based on whether the current LOG_TAG is enabled at
 * verbose priority.
 */
    #ifndef IF_ALOGV
        #if LOG_NDEBUG
            #define IF_ALOGV() if (false)
        #else
            #define IF_ALOGV() IF_ALOG(LOG_VERBOSE, LOG_TAG)
        #endif
    #endif

    /*
 * Conditional based on whether the current LOG_TAG is enabled at
 * debug priority.
 */
    #ifndef IF_ALOGD
        #define IF_ALOGD() IF_ALOG(LOG_DEBUG, LOG_TAG)
    #endif

    /*
 * Conditional based on whether the current LOG_TAG is enabled at
 * info priority.
 */
    #ifndef IF_ALOGI
        #define IF_ALOGI() IF_ALOG(LOG_INFO, LOG_TAG)
    #endif

    /*
 * Conditional based on whether the current LOG_TAG is enabled at
 * warn priority.
 */
    #ifndef IF_ALOGW
        #define IF_ALOGW() IF_ALOG(LOG_WARN, LOG_TAG)
    #endif

    /*
 * Conditional based on whether the current LOG_TAG is enabled at
 * error priority.
 */
    #ifndef IF_ALOGE
        #define IF_ALOGE() IF_ALOG(LOG_ERROR, LOG_TAG)
    #endif

    // ---------------------------------------------------------------------

    /*
 * Simplified macro to send a verbose system log message using the current LOG_TAG.
 */
    #ifndef SLOGV
        #define __SLOGV(...) \
            ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_VERBOSE, LOG_TAG, __VA_ARGS__))
        #if LOG_NDEBUG
            #define SLOGV(...)                \
                do {                          \
                    if (0) {                  \
                        __SLOGV(__VA_ARGS__); \
                    }                         \
                } while (0)
        #else
            #define SLOGV(...) __SLOGV(__VA_ARGS__)
        #endif
    #endif

    #ifndef SLOGV_IF
        #if LOG_NDEBUG
            #define SLOGV_IF(cond, ...) ((void)0)
        #else
            #define SLOGV_IF(cond, ...)                                                                          \
                ((__predict_false(cond))                                                                         \
                     ? ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_VERBOSE, LOG_TAG, __VA_ARGS__)) \
                     : (void)0)
        #endif
    #endif

    /*
 * Simplified macro to send a debug system log message using the current LOG_TAG.
 */
    #ifndef SLOGD
        #define SLOGD(...) \
            ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_DEBUG, LOG_TAG, __VA_ARGS__))
    #endif

    #ifndef SLOGD_IF
        #define SLOGD_IF(cond, ...)                                                                        \
            ((__predict_false(cond))                                                                       \
                 ? ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_DEBUG, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif

    /*
 * Simplified macro to send an info system log message using the current LOG_TAG.
 */
    #ifndef SLOGI
        #define SLOGI(...) \
            ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_INFO, LOG_TAG, __VA_ARGS__))
    #endif

    #ifndef SLOGI_IF
        #define SLOGI_IF(cond, ...)                                                                       \
            ((__predict_false(cond))                                                                      \
                 ? ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_INFO, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif

    /*
 * Simplified macro to send a warning system log message using the current LOG_TAG.
 */
    #ifndef SLOGW
        #define SLOGW(...) \
            ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_WARN, LOG_TAG, __VA_ARGS__))
    #endif

    #ifndef SLOGW_IF
        #define SLOGW_IF(cond, ...)                                                                       \
            ((__predict_false(cond))                                                                      \
                 ? ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_WARN, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif

    /*
 * Simplified macro to send an error system log message using the current LOG_TAG.
 */
    #ifndef SLOGE
        #define SLOGE(...) \
            ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_ERROR, LOG_TAG, __VA_ARGS__))
    #endif

    #ifndef SLOGE_IF
        #define SLOGE_IF(cond, ...)                                                                        \
            ((__predict_false(cond))                                                                       \
                 ? ((void)__android_log_buf_print(LOG_ID_SYSTEM, ANDROID_LOG_ERROR, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif

#endif /* !LINT_RLOG */

// ---------------------------------------------------------------------

/*
 * Simplified macro to send a verbose radio log message using the current LOG_TAG.
 */
#ifndef RLOGV
    #define __RLOGV(...) \
        ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_VERBOSE, LOG_TAG, __VA_ARGS__))
    #if LOG_NDEBUG
        #define RLOGV(...)                \
            do {                          \
                if (0) {                  \
                    __RLOGV(__VA_ARGS__); \
                }                         \
            } while (0)
    #else
        #define RLOGV(...) __RLOGV(__VA_ARGS__)
    #endif
#endif

#ifndef RLOGV_IF
    #if LOG_NDEBUG
        #define RLOGV_IF(cond, ...) ((void)0)
    #else
        #define RLOGV_IF(cond, ...)                                                                         \
            ((__predict_false(cond))                                                                        \
                 ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_VERBOSE, LOG_TAG, __VA_ARGS__)) \
                 : (void)0)
    #endif
#endif

/*
 * Simplified macro to send a debug radio log message using the current LOG_TAG.
 */
#ifndef RLOGD
    #define RLOGD(...) \
        ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_DEBUG, LOG_TAG, __VA_ARGS__))
#endif

#ifndef RLOGD_IF
    #define RLOGD_IF(cond, ...)                                                                       \
        ((__predict_false(cond))                                                                      \
             ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_DEBUG, LOG_TAG, __VA_ARGS__)) \
             : (void)0)
#endif

/*
 * Simplified macro to send an info radio log message using the current LOG_TAG.
 */
#ifndef RLOGI
    #define RLOGI(...) \
        ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_INFO, LOG_TAG, __VA_ARGS__))
#endif

#ifndef RLOGI_IF
    #define RLOGI_IF(cond, ...)                                                                      \
        ((__predict_false(cond))                                                                     \
             ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_INFO, LOG_TAG, __VA_ARGS__)) \
             : (void)0)
#endif

/*
 * Simplified macro to send a warning radio log message using the current LOG_TAG.
 */
#ifndef RLOGW
    #define RLOGW(...) \
        ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_WARN, LOG_TAG, __VA_ARGS__))
#endif

#ifndef RLOGW_IF
    #define RLOGW_IF(cond, ...)                                                                      \
        ((__predict_false(cond))                                                                     \
             ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_WARN, LOG_TAG, __VA_ARGS__)) \
             : (void)0)
#endif

/*
 * Simplified macro to send an error radio log message using the current LOG_TAG.
 */
#ifndef RLOGE
    #define RLOGE(...) \
        ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_ERROR, LOG_TAG, __VA_ARGS__))
#endif

#ifndef RLOGE_IF
    #define RLOGE_IF(cond, ...)                                                                       \
        ((__predict_false(cond))                                                                      \
             ? ((void)__android_log_buf_print(LOG_ID_RADIO, ANDROID_LOG_ERROR, LOG_TAG, __VA_ARGS__)) \
             : (void)0)
#endif

// ---------------------------------------------------------------------

/*
 * Log a fatal error.  If the given condition fails, this stops program
 * execution like a normal assertion, but also generating the given message.
 * It is NOT stripped from release builds.  Note that the condition test
 * is -inverted- from the normal assert() semantics.
 */
#ifndef LOG_ALWAYS_FATAL_IF
#if CC_PLATFORM == CC_PLATFORM_ANDROID
    #define LOG_ALWAYS_FATAL_IF(cond, ...)                                \
        ((__predict_false(cond))                                          \
             ? ((void)android_printAssert(#cond, LOG_TAG, ##__VA_ARGS__)) \
             : (void)0)
#elif CC_PLATFORM == CC_PLATFORM_OPENHARMONY
    #define LOG_ALWAYS_FATAL_IF(cond, ...) ((void)0 )
#endif
#endif

#ifndef LOG_ALWAYS_FATAL
#if CC_PLATFORM == CC_PLATFORM_ANDROID
    #define LOG_ALWAYS_FATAL(...) \
        (((void)android_printAssert(NULL, LOG_TAG, ##__VA_ARGS__)))
#elif CC_PLATFORM == CC_PLATFORM_OPENHARMONY
    #define LOG_ALWAYS_FATAL(...) ((void) OH_LOG_Print(LOG_APP, LOG_ERROR, LOG_DOMAIN, "HMG_LOG", __VA_ARGS__))
#endif
#endif

/*
 * Versions of LOG_ALWAYS_FATAL_IF and LOG_ALWAYS_FATAL that
 * are stripped out of release builds.
 */
#if LOG_NDEBUG

    #ifndef LOG_FATAL_IF
        #define LOG_FATAL_IF(cond, ...) ((void)0)
    #endif
    #ifndef LOG_FATAL
        #define LOG_FATAL(...) ((void)0)
    #endif

#else

    #ifndef LOG_FATAL_IF
        #define LOG_FATAL_IF(cond, ...) LOG_ALWAYS_FATAL_IF(cond, ##__VA_ARGS__)
    #endif
    #ifndef LOG_FATAL
        #define LOG_FATAL(...) LOG_ALWAYS_FATAL(__VA_ARGS__)
    #endif

#endif

/*
 * Assertion that generates a log message when the assertion fails.
 * Stripped out of release builds.  Uses the current LOG_TAG.
 */
#ifndef ALOG_ASSERT
    #define ALOG_ASSERT(cond, ...) LOG_FATAL_IF(!(cond), ##__VA_ARGS__)
//#define ALOG_ASSERT(cond) LOG_FATAL_IF(!(cond), "Assertion failed: " #cond)
#endif

// ---------------------------------------------------------------------

/*
 * Basic log message macro.
 *
 * Example:
 *  ALOG(LOG_WARN, NULL, "Failed with error %d", errno);
 *
 * The second argument may be NULL or "" to indicate the "global" tag.
 */
#ifndef ALOG
#if CC_PLATFORM == CC_PLATFORM_ANDROID
#define ALOG(priority, tag, ...) \
        LOG_PRI(ANDROID_##priority, tag, __VA_ARGS__)
#elif CC_PLATFORM == CC_PLATFORM_OPENHARMONY
#define ALOG(priority, tag, ...) \
        LOG_PRI(priority, tag, __VA_ARGS__) 
#endif
#endif


/*
 * Log macro that allows you to specify a number for the priority.
 */
#ifndef LOG_PRI
#if CC_PLATFORM == CC_PLATFORM_ANDROID
    #define LOG_PRI(priority, tag, ...) \
        android_printLog(priority, tag, __VA_ARGS__)
#elif CC_PLATFORM == CC_PLATFORM_OPENHARMONY
    #define LOG_PRI(priority, tag, ...) ((void) OH_LOG_Print(LOG_APP, priority, LOG_DOMAIN, "HMG_LOG", __VA_ARGS__))
#endif
#endif

/*
 * Log macro that allows you to pass in a varargs ("args" is a va_list).
 */
#ifndef LOG_PRI_VA
    #define LOG_PRI_VA(priority, tag, fmt, args) \
        android_vprintLog(priority, NULL, tag, fmt, args)
#endif

/*
 * Conditional given a desired logging priority and tag.
 */
#ifndef IF_ALOG
    #define IF_ALOG(priority, tag) \
        if (android_testLog(ANDROID_##priority, tag))
#endif

// ---------------------------------------------------------------------

/*
 * ===========================================================================
 *
 * The stuff in the rest of this file should not be used directly.
 */

#define android_printLog(prio, tag, ...) \
    __android_log_print(prio, tag, __VA_ARGS__)

#define android_vprintLog(prio, cond, tag, ...) \
    __android_log_vprint(prio, tag, __VA_ARGS__)

/* XXX Macros to work around syntax errors in places where format string
 * arg is not passed to ALOG_ASSERT, LOG_ALWAYS_FATAL or LOG_ALWAYS_FATAL_IF
 * (happens only in debug builds).
 */

/* Returns 2nd arg.  Used to substitute default value if caller's vararg list
 * is empty.
 */
#define __android_second(dummy, second, ...) second

/* If passed multiple args, returns ',' followed by all but 1st arg, otherwise
 * returns nothing.
 */
#define __android_rest(first, ...) , ##__VA_ARGS__

#define android_printAssert(cond, tag, ...) \
    __android_log_assert(cond, tag,         \
                         __android_second(0, ##__VA_ARGS__, NULL) __android_rest(__VA_ARGS__))

#define android_writeLog(prio, tag, text) \
    __android_log_write(prio, tag, text)

#define android_bWriteLog(tag, payload, len) \
    __android_log_bwrite(tag, payload, len)
#define android_btWriteLog(tag, type, payload, len) \
    __android_log_btwrite(tag, type, payload, len)

#define android_errorWriteLog(tag, subTag) \
    __android_log_error_write(tag, subTag, -1, NULL, 0)

#define android_errorWriteWithInfoLog(tag, subTag, uid, data, dataLen) \
    __android_log_error_write(tag, subTag, uid, data, dataLen)

/*
 *    IF_ALOG uses android_testLog, but IF_ALOG can be overridden.
 *    android_testLog will remain constant in its purpose as a wrapper
 *        for Android logging filter policy, and can be subject to
 *        change. It can be reused by the developers that override
 *        IF_ALOG as a convenient means to reimplement their policy
 *        over Android.
 */
#if LOG_NDEBUG /* Production */
    #define android_testLog(prio, tag) \
        (__android_log_is_loggable(prio, tag, ANDROID_LOG_DEBUG) != 0)
#else
    #define android_testLog(prio, tag) \
        (__android_log_is_loggable(prio, tag, ANDROID_LOG_VERBOSE) != 0)
#endif

/*
 * Use the per-tag properties "log.tag.<tagname>" to generate a runtime
 * result of non-zero to expose a log. prio is ANDROID_LOG_VERBOSE to
 * ANDROID_LOG_FATAL. default_prio if no property. Undefined behavior if
 * any other value.
 */
int __android_log_is_loggable(int prio, const char *tag, int default_prio);

int __android_log_security(); /* Device Owner is present */

int __android_log_error_write(int tag, const char *subTag, int32_t uid, const char *data,
                              uint32_t dataLen);

/*
 * Send a simple string to the log.
 */
int __android_log_buf_write(int bufID, int prio, const char *tag, const char *text);
int __android_log_buf_print(int bufID, int prio, const char *tag, const char *fmt, ...)
#if defined(__GNUC__)
    __attribute__((__format__(printf, 4, 5)))
#endif
    ;

#ifdef __cplusplus
}
#endif

#endif /* COCOS_CUTILS_LOG_H */
